;
;    LPR2DMX: convert printer data to dmx512 using a pic16f84a
;    Copyright (C) 2000 Jan Menzel
;
;    This program is free software; you can redistribute it and/or modify
;    it under the terms of the GNU General Public License as published by
;    the Free Software Foundation; either version 2 of the License, or
;    (at your option) any later version.
;
;    This program is distributed in the hope that it will be useful,
;    but WITHOUT ANY WARRANTY; without even the implied warranty of
;    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;    GNU General Public License for more details.
;
;    You should have received a copy of the GNU General Public License
;    along with this program; if not, write to the Free Software
;    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
; 
;
;
; Lighting-Solutions
; Jan Menzel
; Thadenstr. 80
; 22767 Hamburg
; mailto: jan@lighting-solutions.de
; http://www.lighting-solutions.de
;
;   This DMX512 parallel-port interface converts data received from the parallel-port into DMX512
; signals. On the parallel-port the interface uses EPP/SPP transfer mode using the lines /BUSY, 
; STROBE and ACKNOWLEDGE. The /INIT line is also connected to allow the interface being reseted
; via parallel-port.
;   The interface expects 3 bytes in the beginning: 2 for configuration and the start-code. The
; first configuration byte (the MSB) consists of settings (bits 1-7) and the MSB from the slots
; counter. The second one the LSB from the slots counter. The slots counter is actually the number
; of slots to transmit decreased by one. So a slot counter of 0 means one slot is to be transmitted.
; The maximum number of 512 slots can be transmitted by setting the counter in the configuration
; bytes to 511. After receiving the start-code as many bytes as in the slot counter (+1) are
; expected from the printer-port. The interface then expects an other configuration bytes. Data
; is only transmitted out on the DMX512-line while data is received from the printer-port. If no
; data is ready on the port, no data will be transmitted. There is currently no buffering imple-
; mented. Using the DMX4Linux drivers by Michael Stickel you can get about 37 512-byte-frame/sec 
; without problem or any tweaking.
;   With the settings bits in the upper configuration byte you can change the behaviour of the 
; interface: with the upper bit (7) set HIGH the interface will send a saver-dmx check-sum
; after transmitting the last slot (see http://www.soundlight.de for details on saver-dmx). If
; the 6th bit is set HIGH, the interface will produce only a 4us MARK-after-BREAK as specified in
; the DMX512-1990 specs.
;   If one sends as first configuration byte a -1 just after resetting the interface (with the
; /INIT line of the parallel port), the interface will output user informations using a serial
; protocol with BUSY and STROBE. All data is transfered in bytes with LSB first. The first two
; bytes are the device ID, followed by a 0 terminated string with user informations. A value of 10
; is used as line deliminator. After reading this data the device needs to be reseted via /INIT 
; line.
;
; designed clock-speed: 10MHz
;
; Changes:
;   ??      : erste Version fertig gestellt 
;   17.03.00: - most timing stuff updated to use TIMER0
;             - "trigger" removed
;             Version: 02
;   18.03.00: - user info read back using serial protocol with strobe 
;               and busy. setting bit 2 of the first byte will result in
;               user info if send just after resetting the device. After
;               reading user info the device needs to be reseted again.
;               maximum bit clock speed: 100kHz
;   24.03.00: - user info read back trigger changed to first transmitted
;               byte equal -1
;   31.03.00: - timer-check before transmitting the next frame added
;   11.04.00: v03: due to the read-modify-write behaviour, the state of the
;               output pins must be inited in one step before switching them
;               to outputs
;   16.04.00: v04: acknowledge releasing followed busy to fast.
;   22.05.00: v05: some changes in the timing
;   16.06.00: v06: break-time changes to 120usec 
;   16.07.00: v07: problem with large break time fixed: break time now ~105us
;   13.09.00: v08: - config bit for changing polarity added.
;                  - special start byte for changing break time. After reset, the
;                    CPU expects an other byte after the first config one was -2.
;                    The limits are 0..204 for break_time in usec. Afterwords a
;                    new config cycle beginns.
;   14.11.00: v09: symbole "next" in "change_break_time" was not defined.
;   20.12.01: - all numbers changed to dezimal numbers. The default radix should not matter anymore.
;             v0a finished
;

	list p=16f84a
	include <p16f84a.inc>

	__CONFIG _CP_OFF & _PWRTE_OFF & _WDT_ON & _HS_OSC
	__IDLOCS #v(id)


#define	POS		.0
#define NEG		.1
#define BREAK		.120		; initial break time

; porta: control lines
sout	equ 	.0
ack	equ	.2
busy	equ	.3
strobe	equ	.4

; portb: data

; bits in setting (bits 1-7 from first configuration byte):
saver_dmx	equ	.7	; saver-dmx?
dmx1990		equ	.6	; 4musecs MARK after BREAK?
polarity	equ	.5	; polarity to output

; clock speed for timer calculation
clock	equ	.10	; units MHz
id	equ	0x010a	; ID

	cblock	0x0c

		count		; local counter, used for outputing data to DMX-line
		tmp		; used for outputing data to DMX-line
		countLSB	; main slot counter LSB
		countMSB	; main slot counter MSB

		setting		; setting read as msb from lpr-port
		saver		; for saver-dmx's check-sum
		byte_count	; counter for bytes used in write_userinfos
		break_time	; break_time in usec.
	endc
	
#define BUSY	PORTA, busy
#define STROBE	PORTA, strobe
#define ACK	PORTA, ack

; ********************************************
; * OUT: output data in W to the DMX512-line *
; ********************************************
out:	MACRO	pol
	LOCAL	out_loop, set_bit, out_next

	movwf	tmp			; save value to output

	btfss	INTCON, T0IF		; wait, till timer from last output lint up
	goto	$-1
#if pol == POS
	bcf	PORTA, sout		; output start bit
#else
#if pol == NEG
	bsf	PORTA, sout		; output start bit	
#else
	error	"polarity missing"
#endif
#endif

	movlw	-(.44 *clock/.4-.7)	; prepare timer (44us, offset corrected)
	movwf	TMR0
	bcf	INTCON, T0IF

	movlw	.8			; prepare bit counter
	movwf	count
out_loop:				; output byte (LSB first)
	rrf	tmp, F
	btfsc	STATUS, C
	goto	set_bit
	nop
#if pol == POS
	bcf	PORTA, sout
#else
#if pol == NEG
	bsf	PORTA, sout
#else
	error	"polarity missing"
#endif
#endif
	goto	out_next
set_bit:

#if pol == POS
	bsf	PORTA, sout
#else
#if pol == NEG
	bcf	PORTA, sout
#else
	error	"polarity missing"
#endif
#endif
	goto	$+1
out_next:
	decfsz	count, F
	goto 	out_loop
	goto	$+1
	goto	$+1
	nop
#if pol == POS
	bsf	PORTA, sout		; output 2 stop bits
#else
#if pol == NEG
	bcf	PORTA, sout		; output 2 stop bits
#else
	error	"polarity missing"
#endif
#endif
	ENDM

; *****************************************************************************
; * receive_byte: read one byte from lpr into W, leave the port in busy state *
; *****************************************************************************
receive_byte:	MACRO
	btfsc	STROBE		; wait, till strobe gets active
	goto	$-1

	movf	PORTB, W	; EPP: read value first,
	bsf	BUSY		; then confirm reading with busy
	ENDM

; ****************************************************************************************
; * receive_next_byte: read next byte from lpr into W by first ending the busy state and *
; *   then receiving the next byte via receive_byte macro.                              *
; ****************************************************************************************
receive_next_byte: MACRO
	bcf	ACK		; acknowledge read byte
	btfss	STROBE
	goto	$-1
	bcf	BUSY		; release busy
	nop
	bsf	ACK		; finish acknowledging
	receive_byte		; MACRO: receive next byte into W
	ENDM

; *********************
; * main code segment *
; *********************
reset:	org 	0x00			; first: do the setup

	movlw	#v((1<<sout)+(1<<ack))	; due to read-modify-write, initing
	movwf	PORTA			; the outputs needs to be done in one step

	bsf	STATUS, RP0		; Bank1
	bcf	TRISA, sout
	bcf	TRISA, ack
	bcf	TRISA, busy
	bcf	OPTION_REG, PS0		; set WDT prescaler to 1:64 = 1.15sec
	bcf	OPTION_REG, T0CS	; TIMER0 source: instruction cycle clock

	bcf	STATUS, RP0		; Bank0

	movlw	BREAK			; prepare default break_time
	call	prep_bt

	receive_byte			; MACRO: receive first byte

	movwf	setting
	movwf	tmp
	incf	tmp, F			; received byte == -1?
	btfsc	STATUS, Z
	goto	write_userinfo		; yes? -> write back user information

	incf	tmp, F			; received byte == -2?
	btfsc	STATUS, Z
	call	change_break_time	; yes? -> change break time

	btfsc	setting, polarity
	goto	main_loop1_start
	goto	main_loop0_start

	LOCAL	a = 0
	WHILE	a<.2

main_loop#v(a):
	btfss	INTCON, T0IF		; timer lint up? wait, till next frame can be started
	goto	$-1			; (this timer-check can be moved before setting the output

main_loop#v(a)_start:
#if a == POS
	bcf	PORTA, sout		; start of start condition
#else
#if a == NEG
	bsf	PORTA, sout		; start of start condition
#else
#error	"errors in polarity"
#endif
#endif
	movwf	setting			; save first byte as counter MSB and setting
	andlw	0x01
	movwf	countMSB
	incf	countMSB, F
	
	movf	break_time, W		; set timer to break time
	movwf	TMR0
	bcf	INTCON, T0IF		; clear interrupt bit

	receive_next_byte		; MACRO: receive second configuration byte

	movwf	countLSB		; the second received byte is counter LSB
	incf	countLSB, F		; prepare counter

	clrf	saver			; clean saver-dmx check-sum

	btfss	INTCON, T0IF		; timer lint up first time?
	goto	$-1
	movf	break_time, W
	movwf	TMR0
	bcf	INTCON, T0IF		; clear interrupt bit

	receive_next_byte		; receive start code
	
	clrwdt

	btfss	INTCON, T0IF		; timer lint up second time?
	goto	$-1

#if a == POS
	bsf	PORTA, sout		; start of start condition
#else
	bcf	PORTA, sout		; start of start condition
#endif
	goto	$+1
	nop

	btfsc	setting, dmx1990	; dmx1990?
	goto	n1#v(a)			; no: wait again 4us

	goto	$+1
	goto	$+1
	goto	$+1
	goto	$+1
	goto	$+1
	nop

n1#v(a):
#if a == POS
	out	POS			; MACRO: output start code
#else
	out	NEG
#endif

l1#v(a):
	receive_next_byte		; MACRO: receive as many bytes as listed in countLSB, countMSB 
	
	addwf	saver, F		; calculate saver-dmx check-sum
	
#if a == POS
	out	POS			; MACRO: and output them
#else
	out	NEG
#endif
	decfsz	countLSB, F		; counter
	goto	l1#v(a)
	decfsz	countMSB, F
	goto	l1#v(a)

	movf	saver, W
	btfss	setting, saver_dmx	; saver-dmx?
	goto	main_no_saver#v(a)		

#if a == POS
	out	POS			; MACRO: output check-sum
#else
	out	NEG
#endif
main_no_saver#v(a):
	receive_next_byte		; receive first configuration byte

	goto	main_loop#v(a)		; start again output the next frame
a += 1
	ENDW

; **************************************************************************************
; * change_break_time: read one more byte from parport for break_time and recieve next *
; *   frame start.                                                                     *
; **************************************************************************************
change_break_time:
	receive_next_byte		; MACRO: read next byte from parport
	call	prep_bt			; calculate new break_time

	receive_next_byte		; MACRO: reread first config byte
	movwf	setting
	return

; ******************************************************************************
; * prep_bt: prepare value for break time calculation. The value in break_time *
; *   is loaded twice into timer 0 without wdt.                                *
; ******************************************************************************
prep_bt:
	movwf	tmp
	movwf	break_time
	rrf	tmp, F
	rrf	tmp, W
	andlw	0x3f
	addlw	-.8			; correct timer setups
	addwf	break_time, W
	sublw	0			; invertieren...
	movwf	break_time
	return

; ***********************************************************************
; * write_userinfo: write back info about the interface to the user     *
; *   using strobe as clock line and busy as data. Data is valid while  *
; *   strobe is HIGH, transmission order is always LSB first. After     *
; *   reading this info the device needs to be reseted.                 *
; *   Transmitted data: ID first (16bits), followed by a '0' terminated *
; *   string: "<device name>"<cr>(ascii 13)"<copy right information>".  *
; ***********************************************************************
write_userinfo:
	btfss	PORTA, strobe		; strobe needs to be high before start
	goto	$-1

	movlw	HIGH id			; output id first
	call	write_byte
	movlw	LOW id
	call	write_byte

	clrf	byte_count		; prepare counter for outputting string
	movlw	HIGH string		; prepare table lookup
	movwf	PCLATH
wu_loop:
	movf	byte_count, W
	call	wu_get_next_byte	; read byte
	call	write_byte		; and output it

	movf	byte_count, W
	call	wu_get_next_byte
	movwf	tmp			; check send byte
	btfsc	STATUS, Z		; 0?
	goto	end_write		; finish!
	
	incf	byte_count, F
	goto	wu_loop

end_write:
	clrwdt				; loop till external reset
	goto	$-1

write_byte:				; output the byte in W with LSB first
	movwf	tmp
	movlw	.8
	movwf	count
wb_loop:
	btfsc	PORTA, strobe		; wait till strobe gets low
	goto	$-1

	rrf	tmp, F			; set busy to current bit level
	btfss	STATUS, C
	bcf	PORTA, busy
	btfsc	STATUS, C
	bsf	PORTA, busy
	clrwdt

	btfss	PORTA, strobe		; wait for strobe to go high
	goto	$-1
	
	decfsz	count, F		; loop
	goto	wb_loop

	return

	ORG 0x3B0
wu_get_next_byte:			; lookup-table with information-string
	addwf	PCL, F
string: dt	"LPR2DMX", .10
	dt	"(c) 20.12.2001, Jan Menzel", .10
	dt	"http://www.lighting-solutions.de", 0

	END

